package hirondelle.web4j.util;

import junit.framework.*;
//import junit.ui.TestRunner;
//import junit.textui.TestRunner;
import java.util.regex.*;

/**
 <a href="http://www.junit.org">JUnit</a> test cases for {@link Regex}. 
*/
public final class TESTRegex extends TestCase {

  /**
   Run the test cases.
  */
   public static void main(String args[]) {
     String[] testCaseName = {TESTRegex.class.getName()};
     //Select one of several types of interfaces.
     junit.textui.TestRunner.main(testCaseName);
     //junit.swingui.TestRunner.main(testCaseName);
     //junit.ui.TestRunner.main(testCaseName);
  }

  public TESTRegex(String aName) {
    super(aName);
  }

  // TEST CASES //

  public void testWhitespace() {
    testIsMatch(Regex.WS, "");
    testIsMatch(Regex.WS, " ");
    testIsMatch(Regex.WS, Consts.NEW_LINE);
    testIsMatch(Regex.WS, Consts.TAB);
    
    testIsNotMatch(Regex.WS, "A");
    testIsNotMatch(Regex.WS, "0");
    testIsNotMatch(Regex.WS, "  0 ");
  }

  public void testAnyCharsExceptNewLine(){
    testIsMatch(Regex.ANY_CHARS, "");
    testIsMatch(Regex.ANY_CHARS, " ");
    testIsMatch(Regex.ANY_CHARS, ".");
    testIsMatch(Regex.ANY_CHARS, "A");
    testIsMatch(Regex.ANY_CHARS, "9");
    testIsMatch(Regex.ANY_CHARS, "@#$%@#%@#*%@#(%@)_#(%");
    testIsMatch(Regex.ANY_CHARS, Consts.TAB);
    
    testIsNotMatch(Regex.ANY_CHARS, Consts.NEW_LINE);
  }
  
  public void testStartTag(){
    testIsMatch(Regex.START_TAG, "<");
    
    testIsNotMatch(Regex.START_TAG, "< ");
    testIsNotMatch(Regex.START_TAG, " <");
    testIsNotMatch(Regex.START_TAG, ">");
  }

  public void testEndTag(){
    testIsMatch(Regex.END_TAG, ">");
    testIsMatch(Regex.END_TAG, " >");
    
    testIsNotMatch(Regex.END_TAG, "> ");
    testIsNotMatch(Regex.END_TAG, "<");
  }
  
  public void testAllButEndOfTag(){
    testIsMatch(Regex.ALL_BUT_END_OF_TAG, "");
    testIsMatch(Regex.ALL_BUT_END_OF_TAG, " ");
    testIsMatch(Regex.ALL_BUT_END_OF_TAG, "value='blah'");
    testIsMatch(Regex.ALL_BUT_END_OF_TAG, " value='blah' ");
    testIsMatch(Regex.ALL_BUT_END_OF_TAG, " value=blah ");
    testIsMatch(Regex.ALL_BUT_END_OF_TAG, " value=\"blah\" ");
    testIsMatch(Regex.ALL_BUT_END_OF_TAG, "checked value='blah'");
    testIsMatch(Regex.ALL_BUT_END_OF_TAG, " checked value='blah' ");
    
    testIsNotMatch(Regex.ALL_BUT_END_OF_TAG, "value='blah'>" );
    testIsNotMatch(Regex.ALL_BUT_END_OF_TAG, " checked value='blah' >" );
    testIsNotMatch(Regex.ALL_BUT_END_OF_TAG, " checked value='blah' > " );
  }
  
  public void testQuote(){
    testIsMatch(Regex.QUOTE, "\"");
    testIsMatch(Regex.QUOTE, "'");
    
    testIsNotMatch(Regex.QUOTE, "");
    testIsNotMatch(Regex.QUOTE, " ");
    testIsNotMatch(Regex.QUOTE, "A");
    testIsNotMatch(Regex.QUOTE, "'A'");
    testIsNotMatch(Regex.QUOTE, "\"A\"");
  }
  
  public void testNoSpecialHtmlChars(){
    testIsMatch(Regex.NO_SPECIAL_HTML_CHAR, "p");
    testIsMatch(Regex.NO_SPECIAL_HTML_CHAR, "#");
    testIsMatch(Regex.NO_SPECIAL_HTML_CHAR, " ");
    
    testIsNotMatch(Regex.NO_SPECIAL_HTML_CHAR, "<");
    testIsNotMatch(Regex.NO_SPECIAL_HTML_CHAR, ">");
    testIsNotMatch(Regex.NO_SPECIAL_HTML_CHAR, "'");
    testIsNotMatch(Regex.NO_SPECIAL_HTML_CHAR, "\"");
    testIsNotMatch(Regex.NO_SPECIAL_HTML_CHAR, "blah"); // >1 char
  }
  
  public void testQuotedAttr(){
    testIsMatch(Regex.QUOTED_ATTR, "'blah'");
    testIsMatch(Regex.QUOTED_ATTR, "'blah blah'");
    testIsMatch(Regex.QUOTED_ATTR, "'blah '");
    testIsMatch(Regex.QUOTED_ATTR, "' blah'");
    testIsMatch(Regex.QUOTED_ATTR, "' blah '");
    testIsMatch(Regex.QUOTED_ATTR, "\"blah\"");
    testIsMatch(Regex.QUOTED_ATTR, "\"blah \"");
    testIsMatch(Regex.QUOTED_ATTR, "\"blah blah\"");
    testIsMatch(Regex.QUOTED_ATTR, "\" blah \"");
    testIsNotMatch(Regex.QUOTED_ATTR, "blah");
    testIsNotMatch(Regex.QUOTED_ATTR, " blah ");
    testIsNotMatch(Regex.QUOTED_ATTR, "blah blah");
    testIsNotMatch(Regex.QUOTED_ATTR, "'<blah>'");
    testIsNotMatch(Regex.QUOTED_ATTR, "'blah''");
    testGroupOneIsMatch(Regex.QUOTED_ATTR, "'blah'", "blah");
    testGroupOneIsMatch(Regex.QUOTED_ATTR, "' blah'", " blah");
    testGroupOneIsMatch(Regex.QUOTED_ATTR, "'blah '", "blah ");
    testGroupOneIsMatch(Regex.QUOTED_ATTR, "' blah '", " blah ");
    testGroupOneIsMatch(Regex.QUOTED_ATTR, "'blah blah'", "blah blah");
  }
  
  public void testTrimmedText(){
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " blah ", "blah");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, "blah ", "blah");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " blah ", "blah");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " blah blah ", "blah blah");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, "#", "#");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " #", "#");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " # ", "#");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, "this is good", "this is good");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " this is good", "this is good");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, "  this is good", "this is good");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " this is good ", "this is good");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " this is good  ", "this is good");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, " this is good  ", "this is good");
    testGroupOneIsMatch(Regex.TRIMMED_TEXT, "  this is good ", "this is good");
  }
  
  public void testAttrValue(){
    testIsMatch(Regex.UNQUOTED_ATTR, "blah");
    testIsMatch(Regex.UNQUOTED_ATTR, "-_.:blah");
    testIsMatch(Regex.UNQUOTED_ATTR, "BLAH");
    testIsMatch(Regex.UNQUOTED_ATTR, "0Blah9");
    testIsNotMatch(Regex.UNQUOTED_ATTR, " ");
    testIsNotMatch(Regex.UNQUOTED_ATTR, "");
    testIsNotMatch(Regex.UNQUOTED_ATTR, "blah!");
    testIsNotMatch(Regex.UNQUOTED_ATTR, "blah{");
    testIsNotMatch(Regex.UNQUOTED_ATTR, "blah*");
    
    testIsMatch(Regex.SINGLE_QUOTED_ATTR, "'blah'");
    testIsMatch(Regex.SINGLE_QUOTED_ATTR, "''");
    testIsMatch(Regex.SINGLE_QUOTED_ATTR, "'blah blah < > '");
    testIsMatch(Regex.SINGLE_QUOTED_ATTR, "'This is not a \"bongo\" drum.'");
    testIsMatch(Regex.SINGLE_QUOTED_ATTR, EscapeChars.forRegex("'!@#$!@#$%@$^&*()'") );
    testIsNotMatch(Regex.SINGLE_QUOTED_ATTR, "'This isn't a match'");
    
    testIsMatch(Regex.DOUBLE_QUOTED_ATTR, "\"blah\"");
    testIsMatch(Regex.DOUBLE_QUOTED_ATTR, "\"\"");
    testIsMatch(Regex.DOUBLE_QUOTED_ATTR, "\"This is Billy's bongo <drum>.\"");
    testIsMatch(Regex.DOUBLE_QUOTED_ATTR, "\"This is  <> :; - . _ ! @ \"");
    testIsMatch(Regex.DOUBLE_QUOTED_ATTR, "\"This is a 'so-called' quote.\"");
    testIsNotMatch(Regex.DOUBLE_QUOTED_ATTR, "\"This is not a \"so-called\" match.\"");
    
    testIsMatch(Regex.ATTR_VALUE, "\"blah\"");
    testIsMatch(Regex.ATTR_VALUE, "\"This is Billy's bongo.\"");
    testIsMatch(Regex.ATTR_VALUE, "\"This is a 'so-called' quote.\"");
    testIsMatch(Regex.ATTR_VALUE, "\"This is a ! (thing) of 2.5.% _- , !?:;{}\"");
    testIsMatch(Regex.ATTR_VALUE, "\"\"");
    
    testIsMatch(Regex.ATTR_VALUE, "'blah'");
    testIsMatch(Regex.ATTR_VALUE, "''");
    testIsMatch(Regex.ATTR_VALUE, "'This is \"quoted\" like so.'");
    testIsMatch(Regex.ATTR_VALUE, "'This is a ! (thing) of 2.5.% _- , !?:;{}'");
    
    testIsMatch(Regex.ATTR_VALUE, "blah");
    testIsMatch(Regex.ATTR_VALUE, "-._:BLAH09");
  }
  
  public void testCommaInsertion(){
    testCommas("1", "1");
    testCommas("10", "10");
    testCommas("100", "100");
    testCommas("1000", "1,000");
    testCommas("10000", "10,000");
    testCommas("100000", "100,000");
    testCommas("1000000", "1,000,000");
    testCommas("0", "0");
    testCommas("0.0", "0.0");
    testCommas("0.01", "0.01");
    testCommas("0.001", "0.001");
    testCommas("100000.001", "100,000.001");
  }
  
  public void testEmailAddress(){
    testIsMatch(Regex.EMAIL_ADDR, "bob@blah.com");
    testIsMatch(Regex.EMAIL_ADDR, "bob@blah.blah.com");
    testIsMatch(Regex.EMAIL_ADDR, "bob-.@blah.com");
    testIsMatch(Regex.EMAIL_ADDR, "bob@blah.org");
    testIsMatch(Regex.EMAIL_ADDR, "bob@blah.aero");
    testIsMatch(Regex.EMAIL_ADDR, "bob@blah.uk");
    testIsMatch(Regex.EMAIL_ADDR, "bob@blah.boo.uk");
    
    testIsNotMatch(Regex.EMAIL_ADDR, "-.bob@blah.com");
    testIsNotMatch(Regex.EMAIL_ADDR, "bob");
    testIsNotMatch(Regex.EMAIL_ADDR, "");
    testIsNotMatch(Regex.EMAIL_ADDR, " ");
    testIsNotMatch(Regex.EMAIL_ADDR, "bob@");
    testIsNotMatch(Regex.EMAIL_ADDR, "bob@blah");
    testIsNotMatch(Regex.EMAIL_ADDR, "bob@blah.");
    testIsNotMatch(Regex.EMAIL_ADDR, "bob@blah.joe");
  }
  
  public void testIpAddress(){
    testIsMatch(Regex.IP_ADDR, "0.0.0.0");
    testIsMatch(Regex.IP_ADDR, "10.0.0.0");
    testIsMatch(Regex.IP_ADDR, "100.0.0.0");
    testIsMatch(Regex.IP_ADDR, "255.0.0.0");
    testIsMatch(Regex.IP_ADDR, "219.64.54.0");
    testIsMatch(Regex.IP_ADDR, "127.0.0.1");
    testIsMatch(Regex.IP_ADDR, "255.255.255.255");
    testIsMatch(Regex.IP_ADDR, "220.230.240.250");
    
    testIsNotMatch(Regex.IP_ADDR, "");
    testIsNotMatch(Regex.IP_ADDR, " ");
    testIsNotMatch(Regex.IP_ADDR, "1");
    testIsNotMatch(Regex.IP_ADDR, "255");
    testIsNotMatch(Regex.IP_ADDR, "255.");
    testIsNotMatch(Regex.IP_ADDR, "0.0.0.");
    testIsNotMatch(Regex.IP_ADDR, "0.0.0.0.");
    testIsNotMatch(Regex.IP_ADDR, ".0.0.0.0");
  }
  
  public void testNumber(){
    testIsMatch(Regex.NUMBER, "0");
    testIsMatch(Regex.NUMBER, "123");
    testIsMatch(Regex.NUMBER, "+123");
    testIsMatch(Regex.NUMBER, "-123");
    testIsMatch(Regex.NUMBER, "3.14159");
    testIsMatch(Regex.NUMBER, "-3.14159");
    testIsMatch(Regex.NUMBER, "+0");
    testIsMatch(Regex.NUMBER, "-0");
    testIsMatch(Regex.NUMBER, "+0.0");
    testIsMatch(Regex.NUMBER, "-0.0");
    testIsMatch(Regex.NUMBER, ".2310");
    testIsMatch(Regex.NUMBER, "-.2310");
    testIsMatch(Regex.NUMBER, "+.235");
    testIsMatch(Regex.NUMBER, "+0.2561");
    testIsMatch(Regex.NUMBER, "-0.21365");
    testIsMatch(Regex.NUMBER, "2.");
    
    testIsNotMatch(Regex.NUMBER, "");
    testIsNotMatch(Regex.NUMBER, " ");
    testIsNotMatch(Regex.NUMBER, "A");
    testIsNotMatch(Regex.NUMBER, "A123");
    testIsNotMatch(Regex.NUMBER, "123 ");
    testIsNotMatch(Regex.NUMBER, " 123");
    testIsNotMatch(Regex.NUMBER, "1.2 ");
    testIsNotMatch(Regex.NUMBER, "1,000");
    testIsNotMatch(Regex.NUMBER, "1,000.00");
    testIsNotMatch(Regex.NUMBER, "+ 123");
    testIsNotMatch(Regex.NUMBER, "- 123");
  }

  /*
      * <P>Example matches : <tt>1000, 100.25, .25, 0.25, -.13, -0.13</tt>
   <P>Example mismatches : <tt>1,000.00, 100., 100.0, 100.123, -56.000, .123</tt>

   */
  public void testDollars(){
    testIsMatch(Regex.DOLLARS, "1000");
    testIsMatch(Regex.DOLLARS, "+1000");
    testIsMatch(Regex.DOLLARS, "-1000");
    testIsMatch(Regex.DOLLARS, "100.25");
    testIsMatch(Regex.DOLLARS, "+100.25");
    testIsMatch(Regex.DOLLARS, "-100.25");
    testIsMatch(Regex.DOLLARS, ".25");
    testIsMatch(Regex.DOLLARS, "+.25");
    testIsMatch(Regex.DOLLARS, "-.25");
    testIsMatch(Regex.DOLLARS, "0.25");
    testIsMatch(Regex.DOLLARS, "+0.25");
    testIsMatch(Regex.DOLLARS, "-0.25");
    testIsMatch(Regex.DOLLARS, ".13");
    testIsMatch(Regex.DOLLARS, "+.13");
    testIsMatch(Regex.DOLLARS, "-.13");
    
    testIsNotMatch(Regex.DOLLARS, "1000 ");
    testIsNotMatch(Regex.DOLLARS, " 1000");
    testIsNotMatch(Regex.DOLLARS, "1,000");
    testIsNotMatch(Regex.DOLLARS, "1,0000.00");
    testIsNotMatch(Regex.DOLLARS, "100.");
    testIsNotMatch(Regex.DOLLARS, "100.0");
    testIsNotMatch(Regex.DOLLARS, "100.123");
    testIsNotMatch(Regex.DOLLARS, "-56.000");
    testIsNotMatch(Regex.DOLLARS, ".123");
    testIsNotMatch(Regex.DOLLARS, ".2");
    testIsNotMatch(Regex.DOLLARS, ".222");
    testIsNotMatch(Regex.DOLLARS, ".22A");
  }
  
  public void testMoney(){
    testIsMatch(Regex.MONEY, "1000");
    testIsMatch(Regex.MONEY, "+1000");
    testIsMatch(Regex.MONEY, "-1000");
    testIsMatch(Regex.MONEY, "100.01");
    testIsMatch(Regex.MONEY, "+100.01");
    testIsMatch(Regex.MONEY, "-100.01");
    testIsMatch(Regex.MONEY, "100,12");
    testIsMatch(Regex.MONEY, "100,123");
    testIsMatch(Regex.MONEY, "1000,123");
    testIsMatch(Regex.MONEY, ".253");
    testIsMatch(Regex.MONEY, "0.25");
    testIsMatch(Regex.MONEY, "-.13");
    testIsMatch(Regex.MONEY, "-0.000");
    
    testIsNotMatch(Regex.MONEY, "1,000.00");
    testIsNotMatch(Regex.MONEY, "1.000,00");
    testIsNotMatch(Regex.MONEY, "+1 000");
    //testIsNotMatch(Regex.MONEY, "-1,000"); //passes! maybe this is ok... an English 1,000 is a French 1.000
    //testIsNotMatch(Regex.MONEY, "-1,00"); //passes!
    testIsNotMatch(Regex.MONEY, "-1,000.00");
    testIsNotMatch(Regex.MONEY, "1 000");
    testIsNotMatch(Regex.MONEY, "100.");
    testIsNotMatch(Regex.MONEY, "100,");
    testIsNotMatch(Regex.MONEY, "100.0");
    testIsNotMatch(Regex.MONEY, "100,0");
    testIsNotMatch(Regex.MONEY, "100.1234");
    testIsNotMatch(Regex.MONEY, "-100,1234");
    testIsNotMatch(Regex.MONEY, ",1");
    testIsNotMatch(Regex.MONEY, ",1234");
  }
  
  public void testOrdinal(){
    testIsMatch(Regex.DIGITS, "0");
    testIsMatch(Regex.DIGITS, "1");
    testIsMatch(Regex.DIGITS, "9");
    testIsMatch(Regex.DIGITS, "10");
    testIsMatch(Regex.DIGITS, "11");
    testIsMatch(Regex.DIGITS, "19");
    testIsMatch(Regex.DIGITS, "20");
    testIsMatch(Regex.DIGITS, "1000");
    testIsMatch(Regex.DIGITS, "987988");
    testIsMatch(Regex.DIGITS, "128568521657894");
    testIsMatch(Regex.DIGITS, "01");
    testIsMatch(Regex.DIGITS, "09");
    testIsMatch(Regex.DIGITS, "018");
    testIsMatch(Regex.DIGITS, "0018");
    testIsMatch(Regex.DIGITS, "00018");
    
    testIsNotMatch(Regex.DIGITS, "");
    testIsNotMatch(Regex.DIGITS, " ");
    testIsNotMatch(Regex.DIGITS, "A");
    testIsNotMatch(Regex.DIGITS, "-1");
    testIsNotMatch(Regex.DIGITS, " 1");
    testIsNotMatch(Regex.DIGITS, "1 ");
    testIsNotMatch(Regex.DIGITS, "+1");
    testIsNotMatch(Regex.DIGITS, "-0");
    testIsNotMatch(Regex.DIGITS, "+0");
    testIsNotMatch(Regex.DIGITS, "0.0");
    testIsNotMatch(Regex.DIGITS, "2.0");
    testIsNotMatch(Regex.DIGITS, "27.15");
  }
  
  public void testOrdinalOfLimitedLength(){
    testIsMatch(Regex.forNDigits(1), "0");
    testIsMatch(Regex.forNDigits(1), "1");
    testIsMatch(Regex.forNDigits(1), "9");
    
    testIsMatch(Regex.forNDigits(2), "0");
    testIsMatch(Regex.forNDigits(2), "10");
    testIsMatch(Regex.forNDigits(2), "01");
    testIsMatch(Regex.forNDigits(2), "99");
    
    testIsMatch(Regex.forNDigits(3), "0");
    testIsMatch(Regex.forNDigits(3), "1");
    testIsMatch(Regex.forNDigits(3), "11");
    testIsMatch(Regex.forNDigits(3), "01");
    testIsMatch(Regex.forNDigits(3), "100");
    testIsMatch(Regex.forNDigits(3), "001");
    testIsMatch(Regex.forNDigits(3), "568");
    testIsMatch(Regex.forNDigits(3), "999");
    
    testIsNotMatch(Regex.forNDigits(1), "");
    testIsNotMatch(Regex.forNDigits(1), " ");
    testIsNotMatch(Regex.forNDigits(1), "a");
    testIsNotMatch(Regex.forNDigits(1), "-1");
    testIsNotMatch(Regex.forNDigits(1), "+1");
    testIsNotMatch(Regex.forNDigits(1), "2.0");
    testIsNotMatch(Regex.forNDigits(1), "0.0");
    testIsNotMatch(Regex.forNDigits(1), "01");
    testIsNotMatch(Regex.forNDigits(1), "10");
    
    testIsNotMatch(Regex.forNDigits(2), "a");
    testIsNotMatch(Regex.forNDigits(2), "");
    testIsNotMatch(Regex.forNDigits(2), " ");
    testIsNotMatch(Regex.forNDigits(2), "-1");
    testIsNotMatch(Regex.forNDigits(2), "+1");
    testIsNotMatch(Regex.forNDigits(2), "2.0");
    testIsNotMatch(Regex.forNDigits(2), "0.0");
    testIsNotMatch(Regex.forNDigits(2), "100");
    testIsNotMatch(Regex.forNDigits(2), "001");
    
    testIsNotMatch(Regex.forNDigits(3), "2.0");
    testIsNotMatch(Regex.forNDigits(3), "1000");
    testIsNotMatch(Regex.forNDigits(3), "0111");
    testIsNotMatch(Regex.forNDigits(3), " 111");
    testIsNotMatch(Regex.forNDigits(3), "111 ");
    
    //test that param cannot be less than one
    try {
      Regex.forNDigits(0);
    }
    catch (RuntimeException ex){
      return;
    }
    fail("Should not be able to create regex for ordinal of length 0.");
  }
  
  public void testCamelHump(){
    testCamelHump(fSUCCESS, "LoginName", "Login Name");
    testCamelHump(fSUCCESS, "EmailAddress", "Email Address");
    testCamelHump(fSUCCESS, "BlahBlahBlahBlah", "Blah Blah Blah Blah");
    testCamelHump(fSUCCESS, " LoginName", " Login Name");
    testCamelHump(fSUCCESS, "LoginName ", "Login Name ");
    testCamelHump(fSUCCESS, "Login6Name", "Login6 Name");
    
    testCamelHump(fFAIL, "LoginName", "LoginNAme");
    testCamelHump(fFAIL, "LoginName", "Login NAme");
    testCamelHump(fFAIL, "LoginName", "Login Name ");
    testCamelHump(fFAIL, "LoginName", " Login Name");
  }
  
  public void testSimpleScopedIdentifier(){
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "blah");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "blah42");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "blah_42");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "BlahBlah");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "BLAH_BLAH");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "BLAH_BLAH_42");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "blah.sola");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "TRANSLATE.FETCH_ALL_TRANSLATIONS");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "SECURITY.fetch_users_and_roles");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "SECURITY_BLAH.fetch_users_and_roles");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "SECURITY_42.fetch_users_and_roles_42");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "_");
    testIsMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "b");
    
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "blah.sola.foin");
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "blah..sola");
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, ".blah.sola");
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "blah.sola.");
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "*.blah");
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "blah.*");
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "");
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, " ");
    testIsNotMatch(Regex.SIMPLE_SCOPED_IDENTIFIER, "  ");
  }
  
  public void testHours(){
    testIsMatch(Regex.HOURS, "00");
    testIsMatch(Regex.HOURS, "01");
    testIsMatch(Regex.HOURS, "02");
    testIsMatch(Regex.HOURS, "03");
    testIsMatch(Regex.HOURS, "04");
    testIsMatch(Regex.HOURS, "05");
    testIsMatch(Regex.HOURS, "06");
    testIsMatch(Regex.HOURS, "07");
    testIsMatch(Regex.HOURS, "08");
    testIsMatch(Regex.HOURS, "09");
    testIsMatch(Regex.HOURS, "10");
    testIsMatch(Regex.HOURS, "11");
    testIsMatch(Regex.HOURS, "12");
    testIsMatch(Regex.HOURS, "13");
    testIsMatch(Regex.HOURS, "19");
    testIsMatch(Regex.HOURS, "20");
    testIsMatch(Regex.HOURS, "23");
    
    testIsNotMatch(Regex.HOURS, "0");
    testIsNotMatch(Regex.HOURS, "00 ");
    testIsNotMatch(Regex.HOURS, " 00");
    testIsNotMatch(Regex.HOURS, "12:00");
  }
  
  public void testMinutes(){
    testIsMatch(Regex.MINUTES, "00");
    testIsMatch(Regex.MINUTES, "01");
    testIsMatch(Regex.MINUTES, "09");
    testIsMatch(Regex.MINUTES, "10");
    testIsMatch(Regex.MINUTES, "59");
    
    testIsNotMatch(Regex.MINUTES, "0");
    testIsNotMatch(Regex.MINUTES, "1");
    testIsNotMatch(Regex.MINUTES, "60");
    testIsNotMatch(Regex.MINUTES, "-1");
    testIsNotMatch(Regex.MINUTES, "-00");
    testIsNotMatch(Regex.MINUTES, "100");
  }
  
  public void testHoursAndMinutes(){
    testIsMatch(Regex.HOURS_AND_MINUTES, "00:00");
    testIsMatch(Regex.HOURS_AND_MINUTES, "23:59");
    testIsMatch(Regex.HOURS_AND_MINUTES, "01:00");
    testIsMatch(Regex.HOURS_AND_MINUTES, "00:01");
    testIsMatch(Regex.HOURS_AND_MINUTES, "00:59");
    testIsMatch(Regex.HOURS_AND_MINUTES, "23:00");
    testIsMatch(Regex.HOURS_AND_MINUTES, "23:01");
    testIsMatch(Regex.HOURS_AND_MINUTES, "12:45");
    
    testIsNotMatch(Regex.HOURS_AND_MINUTES, "0:00");
    testIsNotMatch(Regex.HOURS_AND_MINUTES, " 00:00");
    testIsNotMatch(Regex.HOURS_AND_MINUTES, "00:00 ");
    testIsNotMatch(Regex.HOURS_AND_MINUTES, "1:00");
    testIsNotMatch(Regex.HOURS_AND_MINUTES, "3:00");
    testIsNotMatch(Regex.HOURS_AND_MINUTES, "00:60");
    testIsNotMatch(Regex.HOURS_AND_MINUTES, "24:00");
  }
  
  // FIXTURE //

   protected void setUp(){
   }

   protected void tearDown() {
   }

   // PRIVATE  //
   
   private static final boolean fSUCCESS = true;
   private static final boolean fFAIL = false;
   
   private void testIsMatch(String aPattern, String aInput){
     Pattern pattern = Pattern.compile(aPattern);
     Matcher matcher = pattern.matcher(aInput);
     assertTrue(matcher.matches());
   }
   
   private void testIsNotMatch(String aPattern, String aInput){
     Pattern pattern = Pattern.compile(aPattern);
     Matcher matcher = pattern.matcher(aInput);
     assertTrue(!matcher.matches());
   }
   
   private void testGroupOneIsMatch(String aPattern, String aInput, String aGroupOne){
     Pattern pattern = Pattern.compile(aPattern);
     Matcher matcher = pattern.matcher(aInput);
     assertTrue(matcher.matches());
     assertTrue(matcher.group(Regex.FIRST_GROUP).equals(aGroupOne));
   }
   
  private void testCommas(String aText, String aOutput){
    Pattern commas = Pattern.compile(Regex.COMMA_INSERTION);
    Matcher matchForCommas = commas.matcher(aText);
    StringBuffer result = new StringBuffer();
    while ( matchForCommas.find() ){
      matchForCommas.appendReplacement(result, ",");
    }
    matchForCommas.appendTail(result);
    assertTrue(result.toString().equals(aOutput));
  }
  
  private void testCamelHump(boolean aSuccessDesired, String aIn, String aOut){
    String replacement = " $1";
    
    Pattern pattern = Pattern.compile(Regex.CAMEL_HUMP_TEXT);
    Matcher matcher = pattern.matcher(aIn);
    if (aSuccessDesired) {
      assertTrue( matcher.replaceAll(replacement).equals(aOut) );    
    }
    else {
      assertTrue( ! matcher.replaceAll(replacement).equals(aOut) );    
    }
  }
   
}
