package hirondelle.web4j.security;

import hirondelle.web4j.request.RequestParser;

/**
 Determines if a request has an ownership constraint which needs explicit validation for a user id proxy.
 
 <P>This interface addresses the issue of 
 <a href='http://www.owasp.org/index.php/Top_10_2007-A4'>Insecure Direct Object Reference</a>, which 
 is an important security issue for web applications. The issue centers around proper enforcement of 
 <b>data ownership constraints</b>. 
 
 <P>Please see the <a href='http://www.web4j.com/UserGuide.jsp#DataOwnershipConstraints'>User Guide</a> for 
 more information on this important topic.
  
 <a name='UntrustedProxyForUserId'></a>
 <h3>Untrusted Proxy For User Id</h3>
 An untrusted proxy for the user id is defined here as satisfying these two criteria: 
 <ul>
   <li>it's "owned" by a specific user (that is, it has an associated data ownership constraint) 
   <li>it's open to manipulation by the end user (for example, by simply changing a request parameter)
 </ul>
 
 <P>An untrusted identifier typically appears in a link, or in a form's target URL.
 This interface is for defining which requests use an untrusted identifier, and which need to enforce a data 
 ownership constraint in a particular way.
 
 <P>Note that, as explained in the <a href='http://www.web4j.com/UserGuide.jsp#DataOwnershipConstraints'>User Guide</a>, 
 not all data ownership constraints involve an untrusted proxy for the user id - only some do. 
 
 <P>The {@link hirondelle.web4j.Controller} processes each request using your application's configured 
 implementation of this interface. Most applications will likely use the default implementation, 
 {@link hirondelle.web4j.security.UntrustedProxyForUserIdImpl}. 
 The <tt>Controller</tt> logic is roughly as follows:
  <PRE>
get the configured implementation of UntrustedProxyForUserId
if the current request has an untrusted id {
  cast the Action to {@link hirondelle.web4j.security.FetchIdentifierOwner}
  fetch the login name of the user who owns the untrusted id
  compare it to the login name of the current user  
  proceed with the Action only if there is a match
}</PRE>

(Reminder: whenever a user logs in, the login name of the current user is always placed into session scope by the Servlet Container.)
 
 <P>Implementations of this interface will typically extract two items from the underlying request, to determine if the 
 request has an untrusted proxy for the user id :
 <ul>
   <li>the 'noun' - identifies <i>what data</i> is being operated on 
   <li>the 'verb' - what is being <i>done</i> to the data (the operation) 
 </ul>
 
 <P>In some cases, only the noun will be important, since <i>all</i> operations on the data can be restricted to the owner. 
 In other cases, both the noun <i>and</i> the verb will be needed to determine if there is a data ownership constraint.
*/
public interface UntrustedProxyForUserId {
  
  /**
   Returns <tt>true</tt> only if the given request uses an untrusted proxy for the user id.
  */
  boolean usesUntrustedIdentifier(RequestParser aRequestParser);

}
